/*:
 * @target MZ
 * @plugindesc [v1.1] スケーラブルなアイコン制御文字を追加（\IS[id,size], \IF[id]）。必要なら \I を一括縮小も可。
 * @author HS
 *
 * @param iconSpacing
 * @text 文字との間隔(px)
 * @type number
 * @default 4
 *
 * @param baselineOffset
 * @text ベースライン微調整(y)
 * @type number
 * @default 0
 * @desc 行の中央揃えに加えて上下に微調整（+で下へ）
 *
 * @param iconSmooth
 * @text スムージング
 * @type boolean
 * @default true
 * @desc OFFにするとドットがシャープになります（縮小時のにじみ防止）
 *
 * @param overrideI
 * @text 既存 \I[n] を置き換え
 * @type boolean
 * @default false
 * @desc ONにすると \I[n] も defaultIconSize で描画します
 *
 * @param defaultIconSize
 * @text 置き換え時のサイズ(px)
 * @type number
 * @default 24
 * @desc overrideI が ON のときの \I[n] の大きさ
 *
 * @help
 * 使い方：
 *  - \IS[iconId, size]     … sizeは px または %（行高に対する割合）
 *      例) \IS[84,18]   → 18px で描画
 *          \IS[84,75%] → 行の高さの 75% で描画
 *          \IS[84,18,-2] → 第3引数でYオフセット微調整
 *  - \IF[iconId]          … 現在のフォントサイズ（\FS[]）に合わせて描画
 *  - 必要ならプラグインパラメータで \I[n] 全体を縮小（既存イベントはそのまま動作）
 *
 * どこで効く？：Window_Base.drawTextEx を使う全ウィンドウ（メッセージ、選択肢、ヘルプ等）
 *
 * 注意：
 *  - 大きくしすぎると行からはみ出すので、その場合は改行やYオフセットで調整してください。
 *  - 変数に制御文字を入れて使う場合は、イベントのスクリプト欄では
 *    バックスラッシュを2本にして "\\IS[84,18]" のように書いてください。
 */
(() => {
  const PN = "HS_IconScaleMZ";
  const P = PluginManager.parameters(PN);
  const iconSpacing    = Number(P.iconSpacing || 4);
  const baselineOffset = Number(P.baselineOffset || 0);
  const iconSmooth     = String(P.iconSmooth || "true") === "true";
  const overrideI      = String(P.overrideI || "false") === "true";
  const defaultIconSize= Number(P.defaultIconSize || 24);

  // 文字列の [ ... ] をそのまま取得（カンマ区切り対応）
  Window_Base.prototype._hsObtainBracketText = function(textState) {
    const m = /^\[([^\]]*)\]/.exec(textState.text.slice(textState.index));
    if (m) { textState.index += m[0].length; return m[1]; }
    return "";
  };

  // 目的サイズの決定（"18", "75%", "fs" 対応）
  Window_Base.prototype._hsParseSize = function(spec) {
    if (!spec) return 32;
    const s = String(spec).trim().toLowerCase();
    if (s.endsWith("%")) {
      const p = Number(s.slice(0, -1)) || 100;
      return Math.max(1, Math.round(this.lineHeight() * p / 100));
    } else if (s === "fs") {
      return Math.max(1, this.contents.fontSize);
    } else {
      return Math.max(1, Number(s) || 32);
    }
  };

  Window_Base.prototype.drawIconScaled = function(iconIndex, x, y, size) {
    const bitmap = ImageManager.loadSystem("IconSet");
    const pw = ImageManager.iconWidth;
    const ph = ImageManager.iconHeight;
    const sx = (iconIndex % 16) * pw;
    const sy = Math.floor(iconIndex / 16) * ph;
    const old = bitmap.smooth;
    bitmap.smooth = iconSmooth;
    this.contents.blt(bitmap, sx, sy, pw, ph, x, y, size, size);
    bitmap.smooth = old;
  };

  const _proc = Window_Base.prototype.processEscapeCharacter;
  Window_Base.prototype.processEscapeCharacter = function(code, textState) {
    if (code === "IS") { // \IS[id,size(,yofs)]
      const raw = this._hsObtainBracketText(textState);
      const [idStr, sizeStr, yofsStr] = raw.split(",").map(s => s != null ? s.trim() : "");
      const iconId = Number(idStr) || 0;
      const size   = this._hsParseSize(sizeStr);
      const yofs   = Number(yofsStr || 0) + baselineOffset;
      const dy = textState.y + Math.floor((this.lineHeight() - size) / 2) + yofs;
      this.drawIconScaled(iconId, textState.x, dy, size);
      textState.x += size + iconSpacing;
      return;
    } else if (code === "IF") { // \IF[id] → フォントサイズに追従
      const iconId = this.obtainEscapeParam(textState);
      const size   = Math.max(1, this.contents.fontSize);
      const dy = textState.y + Math.floor((this.lineHeight() - size) / 2) + baselineOffset;
      this.drawIconScaled(iconId, textState.x, dy, size);
      textState.x += size + iconSpacing;
      return;
    } else if (overrideI && code === "I") { // 既存 \I を一括置き換え（任意）
      const iconId = this.obtainEscapeParam(textState);
      const size   = Math.max(1, defaultIconSize);
      const dy = textState.y + Math.floor((this.lineHeight() - size) / 2) + baselineOffset;
      this.drawIconScaled(iconId, textState.x, dy, size);
      textState.x += size + iconSpacing;
      return;
    }
    _proc.call(this, code, textState);
  };
})();
